<?php
/**
 * Mobile Joomla! extension
 * https://www.mobilejoomla.com
 *
 * @version    2.1.47
 * @license    GNU/GPL v2 - http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright  (C) 2008-2020 Kuneri Ltd. / Denis Ryabov
 * @date       May 2020
 */
defined('_JEXEC') or die('Restricted access');

class MjQueryBuilder
{
    /** @var JDatabaseDriver|MjDatabaseWrapper */
    protected $dbo;
    /** @var JDatabaseQuery */
    protected $query;
    /** @var bool */
    protected $hasQN;
    /** @var int */
    protected $_limit = 0;

    /**
     * @param JDatabaseDriver|MjDatabaseWrapper $dbo
     */
    public function __construct($dbo)
    {
        $this->dbo = $dbo;
        $this->query = $dbo->getQuery(true);
        $this->hasQN = method_exists($this->query, 'quoteName');
    }

    /**
     * @param string $name
     * @return string
     */
    public function qn($name)
    {
        return $this->hasQN ? $this->query->quoteName($name) : $this->dbo->nameQuote($name);
    }

    /**
     * @param string $value
     * @return string
     */
    public function q($value)
    {
        return $this->dbo->quote($value);
    }

    /**
     * @param string $columns,...
     * @return $this
     */
    public function select($columns)
    {
        foreach (func_get_args() as $column) {
            $this->query->select(strpbrk($column, ' (') === false ? $this->qn($column) : $column);
        }
        return $this;
    }

    /**
     * @param string $table
     * @return $this
     */
    public function delete($table = null)
    {
        $this->query->delete($table);
        return $this;
    }

    /**
     * @param string|array $tables
     * @return $this
     */
    public function insert($tables)
    {
        $this->query->insert($tables);
        return $this;
    }

    /**
     * @param string $table
     * @return $this
     */
    public function update($table)
    {
        $this->query->update(strpos($table, ' ') === false ? $this->qn($table) : $table);
        return $this;
    }

    /**
     * @param string $tables,...
     * @return $this
     */
    public function from($tables)
    {
        foreach (func_get_args() as $table) {
            $this->query->from(strpos($table, ' ') === false ? $this->qn($table) : $table);
        }
        return $this;
    }

    /**
     * @param string $type
     * @param string $conditions
     * @return $this
     */
    public function join($type, $conditions)
    {
        $this->query->join($type, $conditions);
        return $this;
    }

    /**
     * @param string|array $conditions
     * @return $this
     */
    public function innerJoin($conditions)
    {
        return $this->join('INNER', $conditions);
    }

    /**
     * @param string|array $conditions
     * @return $this
     */
    public function outerJoin($conditions)
    {
        return $this->join('OUTER', $conditions);
    }

    /**
     * @param string|array $conditions
     * @return $this
     */
    public function leftJoin($conditions)
    {
        return $this->join('LEFT', $conditions);
    }

    /**
     * @param string|array $conditions
     * @return $this
     */
    public function rightJoin($conditions)
    {
        return $this->join('RIGHT', $conditions);
    }

    /**
     * @param string|array $conditions
     * @param string $glue
     * @return $this
     */
    public function set($conditions, $glue = ',')
    {
        $this->query->set($conditions, $glue);
        return $this;
    }

    /**
     * @param string|array $conditions
     * @param string $glue
     * @return $this
     */
    public function where($conditions, $glue = 'AND')
    {
        $this->query->where($conditions, $glue);
        return $this;
    }

    /**
     * @param string $columns,...
     * @return $this
     */
    public function group($columns)
    {
        foreach (func_get_args() as $column) {
            $this->query->group(strpos($column, ' ') === false ? $this->qn($column) : $column);
        }
        return $this;
    }

    /**
     * @param string|array $conditions
     * @param string $glue
     * @return $this
     */
    public function having($conditions, $glue = 'AND')
    {
        $this->query->having($conditions, $glue);
        return $this;
    }

    /**
     * @param string $columns,...
     * @return $this
     */
    public function order($columns)
    {
        foreach (func_get_args() as $column) {
            $this->query->order(strpos($column, ' ') === false ? $this->qn($column) : $column);
        }
        return $this;
    }

    /**
     * @param string $limit
     * @return $this
     */
    public function limit($limit)
    {
        $this->_limit = $limit;
        return $this;
    }

    /**
     * @return JDatabaseDriver|MjDatabaseWrapper
     */
    public function setQuery()
    {
        $this->dbo->setQuery((string)$this->query, 0, $this->_limit);
        return $this->dbo;
    }

    /**
     * @param string $tables,...
     */
    public function dropTable($tables)
    {
        foreach (func_get_args() as $table) {
            $this->dbo->dropTable($table);
        }
    }

    /**
     * @param string $old
     * @param string $new
     */
    public function renameTable($old, $new)
    {
        $this->dbo->renameTable($old, $new);
    }

    /**
     * @param string $tableName
     * @param array $columns
     * @param array $indices
     * @param array $extra
     * @throws Exception
     */
    public function createTable($tableName, $columns, $indices = array(), $extra = array())
    {
        $config = JFactory::getConfig();
        $driver = $config->get('dbtype', 'mysql');

        switch ($driver) {
            case 'mysql':
            case 'mysqli':
            case 'pdomysql':
            case 'jdiction_mysqli':
                $this->createTableMysql($tableName, $columns, $indices, $extra);
                break;

            case 'postgresql':
            case 'pgsql':
                $this->createTablePostgresql($tableName, $columns, $indices, $extra);
                break;

            default:
                throw new Exception('Unsupported database driver');
        }
    }

    /**
     * @param string $tableName
     * @param array $columns
     * @param array $indices
     * @param array $extra
     * @throws Exception, RuntimeException
     */
    private function createTableMysql($tableName, $columns, $indices, $extra)
    {
        $query = 'CREATE TABLE ';
        if (isset($extra['if_not_exists']) && $extra['if_not_exists'] === true) {
            $query .= 'IF NOT EXISTS ';
        }
        $query .= $this->qn($tableName);
        $query .= '(';
        $buffer = array();
        foreach ($columns as $columnName => $columnType) {
            $q = $this->qn($columnName);
            switch ($columnType['type']) {
                case 'int':
                case 'integer':
                    $q .= ' int';
                    break;
                case 'bigint':
                    $q .= ' int(11)';
                    unset($columnType['size']);
                    break;
                case 'serial':
                    $q .= ' int(10) unsigned auto_increment';
                    unset($columnType['size'], $columnType['unsigned'], $columnType['autoincrement']);
                    break;
                case 'varchar':
                    $q .= ' varchar';
                    break;
                default:
                    throw new Exception('Unsupported field type ' . $columnType['type']);
            }
            if (isset($columnType['size']) && is_int($columnType['size'])) {
                $q .= '(' . $columnType['size'] . ')';
            }
            if (!empty($columnType['unsigned'])) {
                $q .= ' unsigned';
            }
            if (!empty($columnType['notnull'])) {
                $q .= ' not null';
            }
            if (!empty($columnType['autoincrement'])) {
                $q .= ' auto_increment';
            }
            if (isset($columnType['default'])) {
                $q .= ' default ' . $columnType['default'];
            }
            $buffer[] = $q;
        }

        foreach ($indices as $indexName => $columnList) {
            switch ($indexName) {
                case '@primary':
                    $q = array();
                    foreach ($columnList as $columnName) {
                        $q[] = $this->qn($columnName);
                    }
                    $buffer[] = 'PRIMARY KEY (' . implode(',', $q) . ')';
                    break;
                default:
                    $q = array();
                    foreach ($columnList as $columnName) {
                        $q[] = $this->qn($columnName);
                    }
                    $buffer[] = 'KEY ' . $this->qn($indexName) . ' (' . implode(',', $q) . ')';
            }
        }

        $query .= implode(', ', $buffer);
        $query .= ')';
        if (isset($columnType['charset'])) {
            $query .= ' default charset=utf8';
        }
        $this->dbo->setQuery($query);
        $this->dbo->query();
    }

    /**
     * @param string $tableName
     * @param array $columns
     * @param array $indices
     * @param array $extra
     * @throws Exception, RuntimeException
     */
    private function createTablePostgresql($tableName, $columns, $indices, $extra)
    {
        $query = 'CREATE TABLE ';
        $query .= $this->qn($tableName);
        $query .= '(';
        $buffer = array();
        foreach ($columns as $columnName => $columnType) {
            $q = $this->qn($columnName);
            switch ($columnType['type']) {
                case 'int':
                case 'integer':
                    $q .= ' int';
                    break;
                case 'bigint':
                    $q .= ' int(11)';
                    unset($columnType['size']);
                    break;
                case 'serial':
                    $q .= ' serial';
                    unset($columnType['size'], $columnType['unsigned'], $columnType['autoincrement']);
                    break;
                case 'varchar':
                    $q .= ' varchar';
                    break;
                default:
                    throw new Exception('Unsupported field type ' . $columnType['type']);
            }
            if (isset($columnType['size']) && is_int($columnType['size'])) {
                $q .= '(' . $columnType['size'] . ')';
            }
            if (!empty($columnType['unsigned'])) {
                $q .= ' unsigned';
            }
            if (!empty($columnType['autoincrement'])) {
                // @todo Should it be implemented?
            }
            if (isset($columnType['default'])) {
                $q .= ' default ' . $columnType['default'];
            }
            if (!empty($columnType['notnull'])) {
                $q .= ' not null';
            }
            $buffer[] = $q;
        }

        foreach ($indices as $indexName => $columnList) {
            switch ($indexName) {
                case '@primary':
                    $q = array();
                    foreach ($columnList as $columnName) {
                        $q[] = $this->qn($columnName);
                    }
                    $buffer[] = 'PRIMARY KEY (' . implode(',', $q) . ')';
                    break;
            }
        }

        $query .= implode(', ', $buffer);
        $query .= ')';
        $this->dbo->setQuery($query);
        $this->dbo->query();

        foreach ($indices as $indexName => $columnList) {
            switch ($indexName) {
                case '@primary':
                    break;
                default:
                    $q = array();
                    foreach ($columnList as $columnName) {
                        $q[] = $this->qn($columnName);
                    }
                    $query = 'CREATE INDEX ' . $this->qn($indexName) . ' ON ' . $this->qn($tableName) . ' (' . implode(',', $q) . ')';
                    $this->dbo->setQuery($query);
                    $this->dbo->query();
            }
        }
    }
}