<?php
/**
 * Mobile Joomla! extension
 * https://www.mobilejoomla.com
 *
 * @version    2.1.47
 * @license    GNU/GPL v2 - http://www.gnu.org/licenses/gpl-2.0.html
 * @copyright  (C) 2008-2020 Kuneri Ltd. / Denis Ryabov
 * @date       May 2020
 */
defined('_JEXEC') or die('Restricted access');

jimport('joomla.plugin.plugin');
jimport('joomla.environment.uri');
jimport('joomla.filesystem.file');

class plgSystemMobileJoomla extends JPlugin
{
    /** @var MjJoomlaWrapper */
    private $joomlaWrapper;

    /** @var MobileJoomla */
    private $mj;

    /** @var bool */
    private $is_joomla15;

    /** @var array */
    private $devices;

    public function __construct(&$subject, $config)
    {
        parent::__construct($subject, $config);
        $this->is_joomla15 = (strncmp(JVERSION, '1.5.', 4) === 0);
    }

    public function onGetMobileJoomla()
    {
        return $this->mj;
    }

    public function onAfterInitialise()
    {
        /** @var JApplicationSite $app */
        $app = JFactory::getApplication();
        if ($app->isAdmin()) {// don't use MobileJoomla in backend
            return;
        }

        include_once JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/legacy/joomlawrapper.php';
        $this->joomlaWrapper = MjJoomlaWrapper::getInstance();

        //load MobileJoomla class
        require_once JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/classes/mobilejoomla.php';

        include_once JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/classes/mjhelper.php';
        $this->devices = MjHelper::getDeviceList();

        /** @var MobileJoomla $mj */
        $mj = new MobileJoomla($this->joomlaWrapper);
        /** @var MjSettingsModel $mjSettings */
        $mjSettings = $mj->settings;

        $this->mj = $mj;

        if ($mjSettings->get('autoupdate', false)) {
            register_shutdown_function(array($this, 'checkNewVersion'));
        }

        JPluginHelper::importPlugin('mobile');
        $app->triggerEvent('onMJInitialise', array($mj));

        if (isset($_COOKIE['mjmarkup']) && ($_SERVER['REQUEST_METHOD'] === 'GET')) {
            $markup = $this->mj->checkMarkup($_COOKIE['mjmarkup']);
            if ($markup !== false) {
                $uri = JUri::getInstance();
                if ($uri->getVar('device', false) === false) {
                    $uri = clone $uri;
                    $uri->setVar('device', $markup);
                    $app->redirect($uri->toString());
                }
            }
        }

        $cached_data = $app->getUserState('mobilejoomla.cache');
        if ($cached_data !== null) {
            $cached_data = gzinflate(base64_decode($cached_data));
            if ($cached_data !== false) {
                $cached_data = @unserialize($cached_data);
            }
        }

        if (is_array($cached_data) && ($cached_data['ua'] === @$_SERVER['HTTP_USER_AGENT'])) {
            $mj->device = $cached_data['device'];
        } else {
            $app->triggerEvent('onDeviceDetection', array($mj));

            // Note: gzipping is necessary to
            // 1) store long User-Agent header (@todo consider to store hash only)
            // 2) support legacy Mobile-Scientia plugin that stores too much data in $mj->device (@todo deprecate in MJ 3.0)
            $gzlevel = 5;
            $cached_data = array('device' => $mj->device, 'ua' => @$_SERVER['HTTP_USER_AGENT']);
            $cached_data = base64_encode(gzdeflate(serialize($cached_data), $gzlevel));
            $app->setUserState('mobilejoomla.cache', $cached_data);
        }

        /** @var MjDevice $mjDevice */
        $mjDevice = $mj->device;

        $mjDevice->markup = $mj->checkMarkup($mjDevice->markup);

        $mjDevice->real_markup = $mjDevice->markup;

        $app->triggerEvent('onAfterDeviceDetection', array($mj));
        $mjDevice->markup = $mj->checkMarkup($mjDevice->markup);

        $markup = $mjDevice->markup;
        $mjDevice->default_markup = $markup;

        //get user choice
        $user_markup = $this->getUserMarkup();
        if ($user_markup !== false) {
            $markup = $user_markup;
        }

        // template preview
        if (isset($_GET['template'])) {
            $getTemplate = $_GET['template'];
        } elseif (isset($_GET['templateStyle']) && is_int($_GET['templateStyle']) && version_compare(JVERSION, '1.7', '>=')) {
            $getTemplate = (int)$_GET['templateStyle'];
        } else {
            $getTemplate = null;
        }
        if ($getTemplate) {
            foreach ($this->devices as $device => $title) {
                if ($getTemplate === $mjSettings->get($device . '.template')) {
                    $markup = $device;
                    break;
                }
            }
        }

        $mjDevice->markup = $markup;

        if (($mjDevice->screenwidth === 0) || ($mjDevice->screenheight === 0)) {
            // @todo iterate installed modes
            switch ($markup) {
                case 'mobile':
                    $mjDevice->screenwidth = 320;
                    $mjDevice->screenheight = 480;
                    break;
                case 'tablet':
                    $mjDevice->screenwidth = 800;
                    $mjDevice->screenheight = 1280;
                    break;
            }
        }

        if ($mjDevice->imageformats === null) {
            switch ($markup) {
                case 'wml':
                    $mjDevice->imageformats = array('wbmp');
                    break;
                case 'chtml':
                    $mjDevice->imageformats = array('gif');
                    break;
                case 'mobile':
                case 'tablet':
                case 'xhtml':
                case 'iphone':
                    $mjDevice->imageformats = array('png', 'gif', 'jpg');
                    break;
            }
        }
        if ($mjDevice->imageformats !== null && count($mjDevice->imageformats) === 0) {
            foreach ($this->devices as $device => $title) {
                $mjSettings->set($device . '.img', 0);
            }
        }

        $app->triggerEvent('onBeforeMobileMarkupInit', array($mj));
        $mjDevice->markup = $mj->checkMarkup($mjDevice->markup);

        $this->updateUserMarkup();

        $app->triggerEvent('onMobileMarkupInit', array($mj));

        $markup = $mjDevice->markup;
        if (empty($markup)) {
            $mjDevice->markup = false;
            return;
        }

        $mj->setMarkup($markup);

        // set headers here to be compatible with System-Cache
        $mj->generator->setHeader();

        if ($mjSettings->get('mobile_sitename')) {
            $this->joomlaWrapper->setConfig('sitename', $mjSettings->get('mobile_sitename'));
        }

        if (!$this->is_joomla15) //Joomla!1.6+
        {
            if (!$mjSettings->get('caching')) {
                $this->joomlaWrapper->setConfig('caching', 0);
            }

            $cachekey = $mj->getCacheKey();

            $registeredurlparams = isset($app->registeredurlparams) ? $app->registeredurlparams : null;
            if (empty($registeredurlparams)) {
                $registeredurlparams = new stdClass();
            }

            $this->setRequestVar('mjcachekey', $cachekey);
            $registeredurlparams->mjcachekey = 'CMD';

            jimport('joomla.environment.uri');
            $uri = JUri::getInstance();
            $this->setRequestVar('mjurlkey', $uri->toString(array('path', 'query')));
            unset($uri);
            $registeredurlparams->mjurlkey = 'STRING';

            $app->registeredurlparams = $registeredurlparams;

            //fix System-Cache plugin in J!3.0-3.7
            if (
                JPluginHelper::isEnabled('system', 'cache') &&
                version_compare(JVERSION, '3.8', '<') &&
                version_compare(JVERSION, '3.0', '>=')
            ) {
                $dispatcher = JEventDispatcher::getInstance();
                $refObj = new ReflectionObject($dispatcher);
                $refProp = $refObj->getProperty('_observers');
                $refProp->setAccessible(true);
                /** @var array $observers */
                $observers = $refProp->getValue($dispatcher);
                foreach ($observers as $object) {
                    if ($object instanceof plgSystemCache) {
                        $object->_cache_key = '~' . $cachekey . '~' . $object->_cache_key;
                    }
                }
            }
        } else { //Joomla!1.5
            if ($mjSettings->get('caching')) {
                /** @var Joomla\Registry\Registry $config */
                $config = JFactory::getConfig();
                $handler = $config->getValue('config.cache_handler', 'file');
                $handler .= '_mj';
                $config->setValue('config.cache_handler', $handler);
                $class = 'JCacheStorage' . ucfirst($handler);
                $path = JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/override/cachestorage_j15/' . $handler . '.php';
                jimport('joomla.cache.storage');
                JLoader::register($class, $path);
            } else {
                $this->joomlaWrapper->setConfig('caching', 0);
                //disable System-Cache plugin
                $dispatcher = JDispatcher::getInstance();
                foreach ($dispatcher->_observers as $index => $object) {
                    if ($object instanceof plgSystemCache) {
                        include_once JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/legacy/cachestub_j15.php';
                        $object->_cache = new _CacheStub_J15();
                        unset($dispatcher->_observers[$index]);
                        break;
                    }
                }
            }
        }

        $router = $app->getRouter();
        $router->attachBuildRule(array($this, 'buildRule'));

        if (!defined('SHMOBILE_MOBILE_TEMPLATE_SWITCHED')) {
            define('SHMOBILE_MOBILE_TEMPLATE_SWITCHED', 1);
        }
    }

    /**
     * @return string|null
     */
    public function onPageCacheGetKey()
    {
        return isset($this->mj) ? $this->mj->getCacheKey() : null;
    }

    /**
     * @param JRouter $router
     * @param JUri $uri
     */
    public function buildRule(&$router, &$uri)
    {
        $mjDevice = $this->mj->device;
        if ($mjDevice->markup != $mjDevice->default_markup) {
            switch ($uri->getVar('format')) {
                case 'feed':
                case 'json':
                case 'xml':
                    return;
            }
            switch ($uri->getVar('type')) {
                case 'rss':
                case 'atom':
                    return;
            }
            if (($router instanceof shRouter || class_exists('Sh404sefClassRouter', false)) &&
                $uri->getVar('Itemid') && count($uri->getQuery(true)) === 2
            ) // check for sh404sef
            {
                $itemid = (int)$uri->getVar('Itemid');
                $app = JFactory::getApplication();
                $menu = $app->getMenu();
                $item = $menu->getItem($itemid);
                $uri->setQuery($item->query);
                $uri->setVar('Itemid', $itemid);
                $uri->setVar('device', $mjDevice->markup);
            } else {
                $uri->setVar('device', $mjDevice->markup);
            }
        }
    }

    /**
     * @throws Exception
     */
    public function onAfterRoute()
    {
        /** @var JApplicationSite $app */
        $app = JFactory::getApplication();
        if ($app->isAdmin()) { // don't use MobileJoomla in backend
            return;
        }

        // don't filter RSS and non-html
        /** @var JDocument $document */
        $document = JFactory::getDocument();
        $format = $document->getType();
        /* @todo refactor JRequest usage */
        $doctype = $this->joomlaWrapper->getRequestVar('type', false);
        if ($doctype === 'rss' || $doctype === 'atom' || $format !== 'html') {
            return;
        }

        // Load config
        $mjSettings = $this->mj->settings;
        $mjDevice = $this->mj->device;

        jimport('joomla.environment.browser');
        $browser = JBrowser::getInstance();
        if (version_compare(JVERSION, '3.0', '<')) {
            $browser->set('_mobile', $mjDevice->markup !== false);
        } else {
            $refObj = new ReflectionObject($browser);
            $refProp = $refObj->getProperty('mobile');
            $refProp->setAccessible(true);
            $refProp->setValue($browser, $mjDevice->markup !== false);
        }

        JPluginHelper::importPlugin('mobile');
        $app->triggerEvent('onMobileAfterRoute', array($this->mj));

        $this->filterExtensions();

        // "Vary" header for proxy/bots
        if ($mjSettings->get('varyua')) {
            JResponse::setHeader('Vary', 'User-Agent');
        }

        // SEO
        $canonical = $this->mj->getCanonicalURI();
        if ($canonical) {
            /** @var JDocumentHtml $document */
            if ($mjSettings->get('canonical')) {
                $document->addHeadLink($canonical, 'canonical');
            }
            if (isset($_GET['device']) && $mjSettings->get('noindex')) {
                $document->setMetaData('robots', 'noindex, nofollow');
            }
        }

        if ($mjDevice->markup === false) { //desktop
            // check homepage
            $menu = $app->getMenu();
            $lang = JFactory::getLanguage();
            if ($menu->getActive() === $menu->getDefault($lang->getTag())) {
                $this->mj->setHome(true);
            }
            return;
        }

        define('_MJ', 1);

        /** @var MjMarkupGenerator $markupGenerator */
        $markupGenerator = $this->mj->generator;

        if (!$this->is_joomla15 && isset($app->registeredurlparams)) //Joomla!1.6+
        {
            $registeredurlparams = $app->registeredurlparams;
            $this->setRequestVar('mjurlkey', null);
            unset($registeredurlparams->mjurlkey);
            $app->registeredurlparams = $registeredurlparams;
        }

        $app->triggerEvent('onMobile', array($this->mj));

        $template = $this->mj->getParam('template');
        $homepage = $this->mj->getParam('homepage');
        $gzip = $this->mj->getParam('gzip');

        if (in_array((string)$this->joomlaWrapper->getRequestInt('Itemid'), (array)$mjSettings->get('nomjitems'), true)) {
            $template = '';
        }

        //Set template
        if (!empty($template)) {
            if ($this->is_joomla15) {
                $app->setUserState('setTemplate', $template);
                $app->setTemplate($template);
            } else {
                $db = $this->joomlaWrapper->getDbo();
                $query = new MjQueryBuilder($db);
                $template_style = $query
                    ->select('params, template')
                    ->from('#__template_styles')
                    ->where($query->qn('client_id') . '=0')
                    ->where($query->qn('id') . '=' . (int)$template)
                    ->setQuery()
                    ->loadObject();
                $params_data = $template_style->params;
                $template = $template_style->template;
                if (empty($params_data)) {
                    $params_data = '{}';
                }
                if (version_compare(JVERSION, '1.7', '>=')) {
                    $app->setTemplate($template, $params_data);
                } elseif (version_compare(JVERSION, '1.6', '>=')) {
                    $app->setTemplate($template);
                    /** @var stdClass $template_obj */
                    $template_obj = $app->getTemplate(true);
                    /** @var Joomla\Registry\Registry {$template_obj->params} */
                    $template_obj->params->loadJSON($params_data);
                }
            }
        }

        // JHtml overrides
        if (version_compare(JVERSION, '3.0', '<')) {
            jimport('joomla.html.html');
            JHtml::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/override/html');
            if (is_dir($dir = JPATH_THEMES . '/' . $template . '/override/html')) {
                JHtml::addIncludePath($dir);
            }
        } else {
            // load email.php only (workaround for new J!3 class loader)
            if (is_file($path = JPATH_THEMES . '/' . $template . '/override/html/email.php')) {
                JLoader::register('JHtmlEmail', $path, true);
            } else {
                JLoader::register('JHtmlEmail', JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/override/html/email.php', true);
            }
        }

        $this->joomlaWrapper->setConfig('gzip', $gzip);

        //Set headers
        JResponse::clearHeaders();
        $markupGenerator->setHeader();

        $isGetRequest = ($_SERVER['REQUEST_METHOD'] === 'GET');

        /** @var JMenu $menu */
        $menu = $app->getMenu();
        $router = $app->getRouter();
        if ($isGetRequest) {
            $Itemid = $this->joomlaWrapper->getRequestInt('Itemid');
            $item = $menu->getItem($Itemid);
            if (is_object($item)) {
                $current = array_merge($item->query, $router->getVars());
            } else {
                $current = $router->getVars();
            }
            if (!isset($current['Itemid'])) {
                $current['Itemid'] = (string)$Itemid;
            }
            unset(
                $current['device']
                , $current['lang']
                , $current['format']
                , $current['no_html']
                , $current['language']
                , $current['tp']
                , $current['template']
                , $current['templateStyle']
                , $current['start']
                , $current['limit'] // fix for sh404sef
                , $current[session_name()]
            );
            if (isset($current['limitstart']) && $current['limitstart'] == 0) {
                unset($current['limitstart']);
            }
        }

        if ($this->is_joomla15) {
            $default = $menu->getDefault();
        } else {
            $lang = JFactory::getLanguage();
            $default = $menu->getDefault($lang->getTag());
        }
        $home = $default->query;
        $home['Itemid'] = $default->id;

        $mj_home = null;
        if (strncmp($homepage, 'index.php?', 10) === 0) {
            parse_str(substr($homepage, 10), $mj_home);
            if (isset($mj_home['Itemid'])) {
                $mj_home_Itemid = (int)$mj_home['Itemid'];
                if ($this->is_joomla15) {
                    $menu->setDefault($mj_home_Itemid);
                } else {
                    $menu->setDefault($mj_home_Itemid, '*');
                }
            }
            if ($isGetRequest && $current == $mj_home) {
                $this->mj->setHome(true);
            }
        }

        if ($isGetRequest && (count($current) === 0 || $current == $home)) {
            $this->mj->setHome(true);
            if ($homepage) {
                if (isset($mj_home_Itemid)) {
                    global $Itemid;
                    $Itemid = $mj_home_Itemid;
                    $menu->setActive($Itemid);
                    if (version_compare(JVERSION, '3.2', '>=')) {
                        $app->input->set('Itemid', $Itemid);
                        $menu->authorise($Itemid);
                    } elseif (!$this->is_joomla15) {
                        $app->authorise($Itemid);
                    } else {
                        $app->authorize($Itemid);
                    }
                }

                $_SERVER['REQUEST_URI'] = JUri::base(true) . '/' . $homepage;
                if ($mj_home !== null) {
                    $_SERVER['QUERY_STRING'] = substr($homepage, 10);
                    foreach ($current as $key => $val) { //clear old variables
                        unset($_REQUEST[$key], $_GET[$key]);
                    }
                    if (version_compare(JVERSION, '3', '<')) {
                        JRequest::set($mj_home, 'get');
                    } else {
                        foreach ($mj_home as $key => $value) {
                            $app->input->set($key, $value);
                        }
                    }
                    $router->setVars($mj_home);
                } else {
                    $url = 'http';
                    $url .= (isset($_SERVER['HTTPS']) && strcasecmp($_SERVER['HTTPS'], 'off')) ? 's' : '';
                    $url .= '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

                    $uri = new JUri($url);
                    $result = $router->parse($uri);
                    if (version_compare(JVERSION, '3', '<')) {
                        JRequest::set($result, 'get');
                    } else {
                        foreach ($result as $key => $value) {
                            $app->input->set($key, $value);
                        }
                    }
                }
            }
        }
    }

    private function getUserMarkup()
    {
        $markup = false;

        if (isset($_GET['device'])) {
            $markup = $_GET['device'];
            $uri = JUri::getInstance();
            $uri->delVar('device');

            if ($markup === 'auto') {
                return false;
            }
            $markup = $this->mj->checkMarkup($markup);
        }

        if ($markup === false && isset($_COOKIE['mjmarkup'])) {
            $markup = $this->mj->checkMarkup($_COOKIE['mjmarkup']);
        }

        return $markup;
    }

    private function updateUserMarkup()
    {
        if (isset($_GET['template']) || isset($_GET['templateStyle'])) {
            return;
        }

        $mjDevice = $this->mj->device;
        $markup = $mjDevice->markup;

        $app = JFactory::getApplication();
        $app->setUserState('mobilejoomla.markup', $markup);

        $mjSettings = $this->mj->settings;
        $cookie_domain = $mjSettings->get('desktop_domain');
        if (strncmp($cookie_domain, 'www.', 4) === 0) {
            $cookie_domain = substr($cookie_domain, 4);
        }
        $cookie_domain = '.' . $cookie_domain;

        if ($markup != $mjDevice->default_markup) {
            $expire = isset($_COOKIE['mjpreview']) ? time()+10*60*1000 : 0;
            setcookie('mjmarkup', $markup ? $markup : 'desktop', $expire, '/', $cookie_domain);
        } elseif (isset($_COOKIE['mjmarkup'])) {
            setcookie('mjmarkup', '', 1, '/', $cookie_domain);
        }
    }

    private function setRequestVar($name, $value = null)
    {
        if (version_compare(JVERSION, '3.0', '>=')) {
            /** @var JApplicationSite $app */
            $app = JFactory::getApplication();
            $app->input->set($name, $value);
        }
        if ($value !== null) {
            $_REQUEST[$name] = $value;
            $GLOBALS['_JREQUEST'][$name] = array('SET.REQUEST' => true);
        } else {
            unset($_REQUEST[$name], $GLOBALS['_JREQUEST'][$name]);
        }
    }

    public function onAfterRender()
    {
        if (!defined('_MJ')) {
            return;
        }

        /** @var JApplicationSite $app */
        $app = JFactory::getApplication();

        if ($this->is_joomla15) {
            $app->setUserState('setTemplate', null);
        }

        if ($this->mj->getParam('httpcaching')) {
            JResponse::allowCache(true);
            JResponse::setHeader('Vary', 'User-Agent, Cookie'); //
        }
        JResponse::setHeader('Cache-Control', 'no-transform');
    }

    private function filterExtensions()
    {
        $mjDevice = $this->mj->device;

        $markup = $mjDevice->markup;
        if (empty($markup)) {
            $markup = 'desktop';
        }

        jimport('joomla.plugins.helper');
        jimport('joomla.application.module.helper');
        $db = $this->joomlaWrapper->getDbo();

        $query = new MjQueryBuilder($db);
        $query
            ->select('p.folder AS ' . $query->qn('type') . ', p.element AS ' . $query->qn('name'))
            ->from($query->qn('#__mj_plugins') . ' AS mj');
        if ($this->is_joomla15) {
            $query->leftJoin($query->qn('#__plugins') . ' AS p ON p.id=mj.id');
        } else {
            $query->leftJoin($query->qn('#__extensions') . ' AS p ON p.extension_id=mj.id');
        }
        $mj_plugins = $query
            ->where('mj.device=' . $query->q($markup))
            ->setQuery()
            ->loadObjectList();

        if (is_array($mj_plugins)) {
            foreach ($mj_plugins as $plugin) {
                $p = JPluginHelper::getPlugin($plugin->type, $plugin->name);
                if (is_object($p)) {
                    $p->type = '_mj_dummy_';
                }
            }
        }


        $query = new MjQueryBuilder($db);
        $mj_modules = $query
            ->select('m.id, m.position')
            ->from($query->qn('#__mj_modules') . ' AS mj')
            ->leftJoin($query->qn('#__modules') . ' AS m ON m.id=mj.id')
            ->where('mj.device=' . $query->q($markup))
            ->setQuery()
            ->loadObjectList();

        $j_modules = array();
        if (is_array($mj_modules)) {
            foreach ($mj_modules as $module) {
                if (!isset($j_modules[$module->position])) {
                    $j_modules[$module->position] = array();
                    $list = JModuleHelper::getModules($module->position);
                    foreach ($list as $item) {
                        $j_modules[$module->position][$item->id] = $item;
                    }
                }
                if (isset($j_modules[$module->position][$module->id])) {
                    $m = $j_modules[$module->position][$module->id];
                    $m->position = $m->module = $m->name = '_mj_dummy_';
                }
            }
        }
    }

    public function onExtensionAfterUpdate()
    {
        JFile::delete(JPATH_ROOT . '/components/com_mobilejoomla/update.dat');
    }

    public function checkNewVersion()
    {
        if (time() < (int)$this->mj->settings->get('autoupdate_next_check')) {
            return;
        }

        // @todo Use Ressio's filelock
        $lockpath = JPATH_ROOT . '/cache/mj/update.lock';
        $lock = @mkdir($lockpath, 0777, true);
        if (!$lock) {
            // check timestamp -> remove dir if 2+ hours difference
            if (@filemtime($lockpath) < time() - 2 * 3600) {
                @rmdir($lockpath);
            }
            return;
        }

        $this->mj->settings->set('autoupdate_next_check', time() + 24 * 60 * 60);
        $this->mj->settings->save();

        $this->installUpdate();

        @rmdir($lockpath);
    }

    private function installUpdate()
    {
        ignore_user_abort(true);
        set_time_limit(200);
        $current_version = '2.1.47';

        $manifest = JPATH_ADMINISTRATOR . '/components/com_mobilejoomla/mobilejoomla.xml';
        $xmlManifest = file_get_contents($manifest);
        $xmlManifest = simplexml_load_string($xmlManifest);

        if ($xmlManifest === false) {
            return;
        }
        $updateServer = (string)$xmlManifest->updateservers->server[0];

        $xml = @file_get_contents($updateServer);
        if ($xml === false) {
            return;
        }

        $xml = simplexml_load_string($xml);
        if ($xml === false) {
            return;
        }

        $new_version = (string)$xml->update->version;

        if (version_compare($current_version, $new_version, '>=')) {
            return;
        }

        $this->joomlaWrapper->loadLanguageFile('com_mobilejoomla', JPATH_ADMINISTRATOR);

        jimport('joomla.installer.helper');
        jimport('joomla.installer.installer');

        // download
        $app = JFactory::getApplication();
        $update = new stdClass;
        $update->url = 'http://www.mobilejoomla.com/latest2.php';
        $app->triggerEvent('onMJBeforeDownload', array($update));
        $packagefile = JInstallerHelper::downloadPackage($update->url);
        if ($packagefile === false) {
            return;
        }

        // unpack
        $packagepath = $this->joomlaWrapper->getConfig('tmp_path') . '/' . $packagefile;
        if (($unpack = JInstallerHelper::unpack($packagepath)) === false) {
            JInstallerHelper::cleanupInstall($packagepath, false);
            return;
        }

        // install
        /** @var array $unpack */
        $extractdir = $unpack['dir'];
        $installer = new JInstaller();
        $updated = $installer->install($extractdir);
        JInstallerHelper::cleanupInstall($packagepath, $extractdir);

        // send notification email(s)
        if ($updated) {
            $mailfrom = $app->getCfg('mailfrom');
            $fromname = $app->getCfg('fromname');
            $siteURL = JUri::base();

            $subject = JText::_('COM_MJ__UPDATE_EMAIL_NOTIFICATION_SUBJECT');
            $message = JText::_('COM_MJ__UPDATE_EMAIL_NOTIFICATION_MESSAGE');

            $subject = str_replace(array('[VERSION]', '[URL]'), array($new_version, $siteURL), $subject);
            $message = str_replace(array('[VERSION]', '[URL]'), array($new_version, $siteURL), $message);

            $db = $this->joomlaWrapper->getDbo();
            $query = new MjQueryBuilder($db);
            $rows = $query
                ->select($this->is_joomla15 ? 'name, email, usertype' : 'id, name, email')
                ->from('#__users')
                ->where($query->qn('sendEmail') . '=1')
                ->where($query->qn('block') . '=0')
                ->setQuery()
                ->loadObjectList();

            // @TODO move to legacy wrappers
            if (version_compare(JVERSION, '2.5', '<')) {
                // Joomla 1.5-1.7
                foreach ($rows as $row) {
                    if ($this->is_joomla15 && strtolower($row->usertype) !== 'super administrator'){
                        continue;
                    }
                    JUtility::sendMail($mailfrom, $fromname, $row->email, $subject, $message);
                }
            } else {
                // Joomla 2.5-3.x
                foreach ($rows as $row) {
                    $user = JFactory::getUser($row->id);
                    if ($user->authorise('core.create', 'com_users')) {
                        $mailer = JFactory::getMailer();
                        $mailer->sendMail($mailfrom, $fromname, $row->email, $subject, $message);
                    }
                }
            }
        }
    }
}
